<?php 
namespace App\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Http\Authentication\AuthenticationUtils;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Doctrine\ORM\EntityManagerInterface;
use App\Repository\UserRepository;
use App\Form\UserForm;
use App\Entity\UserEntity;

#[Route(path: '/user')]
class UserController extends AbstractController
{
    #[Route(path: '/', name: 'user_index', methods: ['GET'])]
    public function index(Request $request): Response 
    {
        if (!$request->getSession()->has('user')) { 
             $this->addFlash('error', 'Authentication required. Please log in.');
            return $this->redirectToRoute('auth_login');
        }

        return $this->render('user/index.html.twig');
    }


    #[Route('/form/{action}/{id}', name: 'user_form', defaults: ['action' => 'new', 'id' => 0])]
    public function userForm(
        string $action, 
        int $id, 
        Request $request, 
        UserRepository $userRepo, 
        EntityManagerInterface $em,
        UserPasswordHasherInterface $hasher
    ): Response {
        if (!$request->getSession()->has('user')) {
            return $this->redirectToRoute('auth_login');
        }

        $user = ($id > 0) ? $userRepo->find($id) : null;
        $isEdit = ($action === 'u' && $user !== null);
        if ($action === 'd') {
            if (!$user) {
                $this->addFlash('error', 'User not found.');
                return $this->redirectToRoute('user_index');
            }

            $user->setIsDeleted(true);
            $em->flush(); 

            $this->addFlash('success', 'User has been deleted.');
            return $this->redirectToRoute('user_index');
        }

        if (!$user) {
            $user = new UserEntity();
        }

        $form = $this->createForm(UserForm::class, $user, [
            'is_new' => !$isEdit
        ]);

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $plainPassword = $form->get('password')->getData();
            
            if (!empty($plainPassword)) {
                $hashedPassword = $hasher->hashPassword($user, $plainPassword);
                $user->setPassword($hashedPassword);
            }

            $em->persist($user);
            $em->flush();

            $this->addFlash('success', 'User saved successfully!');
            return $this->redirectToRoute('user_index');
        }

        return $this->render('user/form.html.twig', [
            'userForm' => $form->createView(),
            'action' => $action,
            'edit_mode' => $isEdit
        ]);
    }

    #[Route('/ajax_list', name: 'user_ajax_list')]
    public function ajaxList(Request $request, UserRepository $userRepository): JsonResponse
    {
        $searchTerm = $request->query->get('q', '');
        $page = $request->query->getInt('page', 1);
        $limit = 10;
        $offset = ($page - 1) * $limit;

        $queryBuilder = $userRepository->createQueryBuilder('u')
        ->where('u.isDeleted = :isDeleted')
        ->setParameter('isDeleted', false)
        ->andWhere('u.type != :superAdmin')
        ->setParameter('superAdmin', 'Super Admin')
        ->andWhere('(u.email LIKE :search OR u.firstName LIKE :search OR u.lastName LIKE :search)')
        ->setParameter('search', '%' . $searchTerm . '%');

        $countQueryBuilder = clone $queryBuilder;

        $users = $queryBuilder
            ->setFirstResult($offset)
            ->setMaxResults($limit)
            ->orderBy('u.lastName', 'ASC')
            ->getQuery()
            ->getResult();

        $totalUsers = $countQueryBuilder
            ->select('count(u.id)')
            ->getQuery()
            ->getSingleScalarResult();

            $data = [];
            foreach ($users as $user) {
                $data[] = [
                    'id' => $user->getId(),
                    'firstName' => $user->getFirstName(),
                    'lastName' => $user->getLastName(),
                    'email' => $user->getEmail(),
                    'type' => $user->getType(),
                ];
            }

            return new JsonResponse([
                'data' => $data,
                'total' => $totalUsers,
                'page' => $page,
                'last_page' => ceil($totalUsers / $limit)
            ]);
    }
}
?>