<?php 
namespace App\Controller;

use Symfony\Component\HttpFoundation\StreamedResponse;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Http\Authentication\AuthenticationUtils;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Doctrine\ORM\EntityManagerInterface;
use App\Repository\SmsRepository;

#[Route(path: '/sms')]
class SmsController extends AbstractController
{
    #[Route(path: '/', name: 'sms_index', methods: ['GET'])]
    public function index(Request $request): Response 
    {
        if (!$request->getSession()->has('user')) { 
             $this->addFlash('error', 'Authentication required. Please log in.');
            return $this->redirectToRoute('auth_login');
        }

        return $this->render('sms/index.html.twig');
    }


    #[Route('/ajax_list', name: 'sms_ajax_list')]
    public function ajaxList(Request $request, SmsRepository $smsRepository): JsonResponse
    {
        $searchTerm = $request->query->get('q', '');
        $status = $request->query->get('status', '');
        $dateFilter = $request->query->get('date', '');
        
        $page = $request->query->getInt('page', 1);
        $limit = 10;
        $offset = ($page - 1) * $limit;

        $queryBuilder = $smsRepository->createQueryBuilder('s');

        if (!empty($searchTerm)) {
            $queryBuilder->andWhere('(s.clientName LIKE :search OR s.companyName LIKE :search)')
                        ->setParameter('search', '%' . $searchTerm . '%');
        }
        if (!empty($status)) {
            $queryBuilder->andWhere('s.status = :status')
                        ->setParameter('status', strtoupper($status)); 
        }

        if (!empty($dateFilter)) {
            $queryBuilder->andWhere('s.dateReturn LIKE :date')
                        ->setParameter('date', $dateFilter . '%');
        }

        $countQueryBuilder = clone $queryBuilder;

        $smsRecords = $queryBuilder
            ->setFirstResult($offset)
            ->setMaxResults($limit)
            ->orderBy('s.id', 'DESC') 
            ->getQuery()
            ->getResult();

        $totalSms = $countQueryBuilder
            ->select('count(s.id)')
            ->getQuery()
            ->getSingleScalarResult();

        $data = [];
        foreach ($smsRecords as $sms) {
            $data[] = [
                'id' => $sms->getId(),
                'companyName' => $sms->getCompanyName(),
                'clientName' => $sms->getClientName(),
                'clientNo' => $sms->getClientNo(),
                'message' => $sms->getMessage(),
                'dateReturn' => $sms->getDateReturn() instanceof \DateTimeInterface 
                    ? $sms->getDateReturn()->format('Y-m-d H:i') 
                    : $sms->getDateReturn(), 
                'senderNo' => $sms->getSenderNo() ?? 'N/A',
                'status' => $sms->getStatus()
            ];
        }

        return new JsonResponse([
            'data' => $data,
            'total' => (int)$totalSms,
            'page' => $page,
            'last_page' => ceil($totalSms / $limit)
        ]);
    }

    #[Route('/sms/export-csv', name: 'sms_export_csv', methods: ['GET'])]
    public function exportCsv(Request $request, SmsRepository $smsRepository): StreamedResponse
    {
        $query = $request->query->get('q');
        $status = $request->query->get('status');
        $date = $request->query->get('date');

        $records = $smsRepository->findForExport($query, $status, $date);

        $response = new StreamedResponse(function () use ($records) {
            $handle = fopen('php://output', 'w+');

            fputcsv($handle, ['Clinic', 'Client Name', 'Phone', 'Message', 'Date Return', 'Sender No', 'Status']);

            foreach ($records as $sms) {
                fputcsv($handle, [
                    $sms->getCompanyName(),
                    $sms->getClientName(),
                    $sms->getClientNo(),
                    $sms->getMessage(),
                    $sms->getDateReturn(),
                    $sms->getSenderNo(),
                    $sms->getStatus()
                ]);
            }

            fclose($handle);
        });

        $response->headers->set('Content-Type', 'text/csv; charset=utf-8');
        $response->headers->set('Content-Disposition', 'attachment; filename="sms_report_'.date('Y-m-d').'.csv"');

        return $response;
    }
}
?>