<?php 
namespace App\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Http\Authentication\AuthenticationUtils;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Doctrine\ORM\EntityManagerInterface;
use App\Repository\ServerRepository;
use App\Form\ServerForm;
use App\Entity\ServerEntity;

#[Route(path: '/server')]
class ServerController extends AbstractController
{
    #[Route(path: '/', name: 'server_index', methods: ['GET'])]
    public function index(Request $request): Response 
    {
        if (!$request->getSession()->has('user')) { 
             $this->addFlash('error', 'Authentication required. Please log in.');
            return $this->redirectToRoute('auth_login');
        }

        return $this->render('server/index.html.twig');
    }

    #[Route('/form/{action}/{id}', name: 'server_form', defaults: ['action' => 'new', 'id' => 0])]
    public function serverForm(
        string $action, 
        int $id, 
        Request $request, 
        ServerRepository $serverRepo, 
        EntityManagerInterface $em
    ): Response {
        if (!$request->getSession()->has('user')) {
            return $this->redirectToRoute('auth_login');
        }

        $server = ($id > 0) ? $serverRepo->find($id) : new ServerEntity();

        if ($action === 'd') {
            if (!$server->getId()) { 
                $this->addFlash('error', 'Server not found.');
                return $this->redirectToRoute('server_index');
            }

            $server->setIsDeleted(true);
            $server->setIsActive(false);
            $em->flush(); 

            $this->addFlash('success', 'Server has been deleted.');
            return $this->redirectToRoute('server_index');
        }

        $isEdit = ($server->getId() !== null);

        $form = $this->createForm(ServerForm::class, $server, [
            'is_new' => !$isEdit
        ]);

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $em->persist($server);
            $em->flush();

            $this->addFlash('success', 'Server saved successfully!');
            return $this->redirectToRoute('server_index');
        }

        return $this->render('server/form.html.twig', [
            'serverForm' => $form->createView(),
            'action' => $action,
            'edit_mode' => $isEdit
        ]);
    }


    #[Route('/ajax_list', name: 'server_ajax_list')]
    public function ajaxList(Request $request, ServerRepository $serverRepository): JsonResponse
    {
        $searchTerm = $request->query->get('q', '');
        $page = $request->query->getInt('page', 1);
        $limit = 10;
        $offset = ($page - 1) * $limit;

        $queryBuilder = $serverRepository->createQueryBuilder('s')
        ->where('s.isDeleted = :isDeleted')
        ->setParameter('isDeleted', false)
        ->andWhere('(s.code LIKE :search OR s.description LIKE :search)')
        ->setParameter('search', '%' . $searchTerm . '%');

        $countQueryBuilder = clone $queryBuilder;

        $servers = $queryBuilder
            ->setFirstResult($offset)
            ->setMaxResults($limit)
            ->orderBy('s.code', 'ASC')
            ->getQuery()
            ->getResult();

        $totalServers = $countQueryBuilder
            ->select('count(s.id)')
            ->getQuery()
            ->getSingleScalarResult();

            $data = [];
            foreach ($servers as $server) {
                $data[] = [
                    'id' => $server->getId(),
                    'code' => $server->getCode(),
                    'description' => $server->getDescription(),
                    'status' => $server->isActive() ? 'Active' : 'InActive',
                ];
            }

            return new JsonResponse([
                'data' => $data,
                'total' => $totalServers,
                'page' => $page,
                'last_page' => ceil($totalServers / $limit)
            ]);
    }
}
?>