<?php 
namespace App\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Http\Authentication\AuthenticationUtils;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Doctrine\ORM\EntityManagerInterface;
use App\Repository\CompanyRepository;
use App\Form\CompanyForm;
use App\Entity\CompanyEntity;

#[Route(path: '/company')]
class CompanyController extends AbstractController
{
    #[Route(path: '/', name: 'company_index', methods: ['GET'])]
    public function index(Request $request): Response 
    {
        if (!$request->getSession()->has('user')) { 
             $this->addFlash('error', 'Authentication required. Please log in.');
            return $this->redirectToRoute('auth_login');
        }

        return $this->render('company/index.html.twig');
    }

 
    #[Route('/ajax_list', name: 'company_ajax_list')]
    public function ajaxList(Request $request, CompanyRepository $companyRepository): JsonResponse
    {
        $searchTerm = $request->query->get('q', '');
        $page = $request->query->getInt('page', 1);
        $limit = 10;
        $offset = ($page - 1) * $limit;

        $queryBuilder = $companyRepository->createQueryBuilder('s')
        ->where('s.isDeleted = :isDeleted')
        ->setParameter('isDeleted', false)
        ->andWhere('(s.code LIKE :search OR s.description LIKE :search)')
        ->setParameter('search', '%' . $searchTerm . '%');

        $countQueryBuilder = clone $queryBuilder;

        $companys = $queryBuilder
            ->setFirstResult($offset)
            ->setMaxResults($limit)
            ->orderBy('s.code', 'ASC')
            ->getQuery()
            ->getResult();

        $totalCompanys = $countQueryBuilder
            ->select('count(s.id)')
            ->getQuery()
            ->getSingleScalarResult();

            $data = [];
            foreach ($companys as $company) {
                $data[] = [
                    'id' => $company->getId(),
                    'code' => $company->getCode(),
                    'description' => $company->getDescription(),
                    'isSmsOnTheDay' => $company->isSmsOnTheDay()
                ];
            }

            return new JsonResponse([
                'data' => $data,
                'total' => $totalCompanys,
                'page' => $page,
                'last_page' => ceil($totalCompanys / $limit)
            ]);
    }

    #[Route('/toggle-ontheday/{id}', name: 'company_toggle_ontheday', methods: ['POST'])]
    public function toggleOnTheDay(int $id, EntityManagerInterface $em): JsonResponse
    {
        $company = $em->getRepository(CompanyEntity::class)->find($id);

        if (!$company) {
            return new JsonResponse(['success' => false, 'message' => 'Company not found'], 404);
        }

        $company->setIsSmsOnTheDay(true);
        
        $em->flush();

        return new JsonResponse(['success' => true]);
    }
}
?>