<?php 
namespace App\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Http\Authentication\AuthenticationUtils;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

use App\Repository\SmsRepository;
use App\Entity\CompanyEntity;


#[Route(path: '/api/v1')]
class ApiController extends AbstractController
{
    #[Route('/sms-tomorrow', name: 'api_sms_tomorrow', methods: ['GET'])]
    public function getTomorrowSchedule(Request $request, SmsRepository $smsRepository, EntityManagerInterface $em): JsonResponse
    {
        $senderNo = $request->query->get('sender_no');
        if (!$senderNo) {
            return new JsonResponse(['status' => 'error', 'message' => 'sender_no required'], 400);
        }

        $tomorrow = new \DateTime('tomorrow');
        $dateStr = $tomorrow->format('Y-m-d');
        $startStr = $dateStr . 'T00:00:00';
        $endStr = $dateStr . 'T23:59:59';

        $queryBuilder = $smsRepository->createQueryBuilder('s')
            ->leftJoin(\App\Entity\CompanyEntity::class, 'c', 'WITH', 'TRIM(s.companyName) = TRIM(c.code)')
            ->where('s.dateReturn >= :start')
            ->andWhere('s.dateReturn <= :end')
            ->andWhere('s.status = :status')
            ->andWhere('(c.id IS NULL OR c.isSmsOnTheDay = :isOnDay)')
            ->setParameter('start', $startStr)
            ->setParameter('end', $endStr)
            ->setParameter('status', 'QUEUED')
            ->setParameter('isOnDay', false) 
            ->orderBy('s.id', 'ASC')
            ->setMaxResults(50);

        $smsRecords = $queryBuilder->getQuery()->getResult();

        if (empty($smsRecords)) {
            return new JsonResponse(['status' => 'success', 'total' => 0, 'data' => []]);
        }

        $output = [];
        foreach ($smsRecords as $sms) {
            $sms->setStatus('SENDING');
            $sms->setSenderNo($senderNo);
            $sms->setSendingAttempt(($sms->getSendingAttempt() ?? 0) + 1);

            $output[] = [
                'id' => (int)$sms->getId(), // Ensure ID is an integer
                'company' => (string)$sms->getCompanyName(),
                'client' => (string)$sms->getClientName(),
                'phone' => (string)$sms->getClientNo(),
                'message' => (string)$sms->getMessage(),
                'time' => $sms->getDateReturn() instanceof \DateTimeInterface 
                        ? $sms->getDateReturn()->format('Y-m-d H:i:s') 
                        : $sms->getDateReturn(), 
            ];
        }

        $em->flush();

        return new JsonResponse([
            'status' => 'success',
            'target' => $dateStr,
            'total' => count($output),
            'data' => $output
        ]);
    }

  #[Route('/sms-confirm', name: 'api_sms_confirm', methods: ['POST'])]
public function confirmSent(Request $request, SmsRepository $smsRepository, EntityManagerInterface $em): JsonResponse
{
    try {
        $content = $request->getContent();
        $data = json_decode($content, true);
        $results = $data['results'] ?? [];

        if (empty($results)) {
            return new JsonResponse(['status' => 'error', 'message' => 'No data'], 400);
        }

        $updatedIds = [];
        foreach ($results as $item) {
            $id = isset($item['id']) ? (int)$item['id'] : null;
            $status = $item['status'] ?? 'SENT';

            if ($id) {
                $sms = $smsRepository->find($id);
                if ($sms) {
                    $sms->setStatus(strtoupper($status));
                    $sms->setUpdatedAt(new \DateTime()); 
                    
                    $updatedIds[] = $id;
                }
            }
        }

        $em->flush();

        return new JsonResponse([
            'status' => 'success',
            'processed_count' => count($updatedIds),
        ]);
        
    } catch (\Exception $e) {
        return new JsonResponse([
            'status' => 'error',
            'message' => $e->getMessage()
        ], 500);
    }
}
}
?>