<?php 

namespace App\Command;

use App\Entity\SmsEntity;
use App\Entity\ServerEntity;
use App\Entity\CompanyEntity;
use App\Repository\ServerRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Contracts\HttpClient\HttpClientInterface;

#[AsCommand(name: 'app:pull-server-data', description: 'Pulls data from active servers in bulk')]
class ServerDataGetterCommand extends Command
{
    public function __construct(
        private ServerRepository $serverRepository,
        private EntityManagerInterface $entityManager,
        private HttpClientInterface $httpClient
    ) {
        parent::__construct();
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $activeServers = $this->serverRepository->findBy(['isActive' => true]);

        if (!$activeServers) {
            $io->info('No active servers found.');
            return Command::SUCCESS;
        }

        $batchSize = 50;
        $count = 0;
        $companyCache = [];

        foreach ($activeServers as $server) {
            $io->note("Pulling data from: " . $server->getCode());

            try {
                
                $baseUrl = $server->getDescription();
                if (empty($baseUrl)) {
                    $io->warning("Server " . $server->getCode() . " has no description/URL.");
                    continue;
                }

                if (!str_starts_with($baseUrl, 'http')) {
                    $baseUrl = 'http://' . $baseUrl;
                }

                $url = rtrim($baseUrl, '/') . '/api/SMSSchedule/PendingOneDayBeforeAppointment';
                
                $response = $this->httpClient->request('POST', $url, ['timeout' => 15]);

                if (200 !== $response->getStatusCode()) {
                    $io->error(sprintf("Server %s returned status %d", $server->getCode(), $response->getStatusCode()));
                    continue; 
                }

                $externalData = $response->toArray();
                $records = $externalData['Records'] ?? [];

                foreach ($records as $recordData) {
                    $uniqueId = $recordData['UniqueID'] ?? null;
                    $companyName = trim($recordData['Name_Company'] ?? 'Unknown Company');

                    if ($uniqueId) {
                        $existingSms = $this->entityManager->getRepository(SmsEntity::class)
                            ->findOneBy(['uniqueId' => $uniqueId]);
                        if ($existingSms) continue;
                    }

                    if (!isset($companyCache[$companyName])) {
                        $company = $this->entityManager->getRepository(CompanyEntity::class)
                            ->findOneBy(['code' => $companyName]);

                        if (!$company) {
                            $company = new CompanyEntity();
                            $company->setCode($companyName);
                            $company->setDescription($companyName);
                            $this->entityManager->persist($company);
                        }
                        $companyCache[$companyName] = $company;
                    }

                    $sms = new SmsEntity(); 
                    $sms->setMessage($recordData['Message'] ?? null);
                    $sms->setClientNo($recordData['ContactNumber_Client'] ?? null);
                    $sms->setCompanyName($companyName);
                    $sms->setClientName($recordData['Name_Client'] ?? null);
                    $sms->setUniqueId($uniqueId);
                    $sms->setDateReturn($recordData['DateReturn'] ?? null);
                    $sms->setStatus('QUEUED');
                    $sms->setServer($server);
                    $sms->setScheduleDateCreated(new \DateTime());

                    $this->entityManager->persist($sms);
                    $count++;

                    if (($count % $batchSize) === 0) {
                        $this->entityManager->flush();
                        $this->entityManager->clear();
                        
                        $server = $this->entityManager->find(ServerEntity::class, $server->getId());
                        $companyCache = []; 
                    }
                }
                
                $this->entityManager->flush();

            } catch (\Exception $e) {
                $io->error("Error on server " . $server->getCode() . ": " . $e->getMessage());
                continue; 
            }
        }

        $io->success("Data pull complete! Imported: $count records.");
        return Command::SUCCESS;
    }
}

?>